/************************************************************************
 * NAME:	fquery
 *
 * DESCR:	Let's you dump certain interesting things out about an
 *		image.
 *
 *		See Usage at bottom.
 *
 * NOTE -
 ************************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>

#include "standard.h"
#include "error.h"
#include "floppy.h"

/************************************************************************/
/*		  RELEASE VERSION NUMBER GOES HERE!			*/
/************************************************************************/

#define VERSION	"2.2"

main(argc,argv)
int	argc;
char	*argv[];
{
    int			ch;
    int			fd;
    struct floppy	floppy;
    int			just_guess = FALSE;	/* don't generate svd	*/
    int			check = FALSE;		/* check CRC		*/
    int			wp = FALSE;		/* write protect	*/
    int			force_CRC = FALSE;	/* force CRC write	*/
    int			verbose = FALSE;	/* level of verbosity	*/
    int			terse = FALSE;		/* level of verbosity	*/
    int			report = FALSE;		/* report disk params	*/
    int			stdoutput = FALSE;	/* use stdout		*/
    int			ignore_CRC = FALSE;	/* ignore CRC errors	*/
    int			machineform = FALSE;	/* machine readable form*/
    int			sector = 0;
    int			track = 0;
    int			dumpphys = TRUE;	/* defaults to phys dump */
    int			dumpsect = FALSE;	/* TRUE to dump sector	*/
    int			finddir = FALSE;	/* find the directory	*/
    char		*format_id = NULL;
    char		*force_format_id = NULL;
    int			inoctal = FALSE;

  (void)floppy_init(&floppy);

  floppy_format_init(); 	/* initialize all supported disk formats */

  opterr = 0;

  for (;;) {
    ch = getopt(argc, argv, "?ogcwCvrs:itmVT:S:LDF");
    if (ch == -1) break;
    switch (ch) {
    case 'o':
	inoctal = TRUE;
	break;
    case 'T':
	track = atoi(argv[optind-1]);
	break;
    case 'S':
	sector = atoi(argv[optind-1]);
	break;
    case 'L':
	dumpphys = FALSE;
	break;
    case 'D':
	dumpsect = TRUE;
	break;
    case 'F':
	finddir = TRUE;
	break;
    case 'i':
	ignore_CRC = TRUE;
	break;
    case 'r':
	report = TRUE;
	break;
    case 'v':
	/* may want to turn this into a number	*/
	verbose = TRUE;
	break;
    case 't':
	terse = TRUE;
	break;
    case 'm':
	machineform = TRUE;
	break;
    case 'w':
	wp = TRUE;
	break;
    case 'c':
	check = TRUE;
	break;
    case 'C':
	force_CRC = TRUE;
	break;
    case 's':
	force_format_id = floppy_format_list_select(argv[optind-1],FALSE);
	if (force_format_id == NULL) {
	    fprintf(stderr,"Format type \"%s\" unknown.\n",argv[optind-1]);
	    usage(argv[0]);
	    exit(3);
	}
	break;
    case 'g':
	just_guess = TRUE;
	break;
    case 'V':
	printf("%s: version %s (%s)\n",argv[0],VERSION,COMPDATE);
	exit(0);
    case '?':
	usage(argv[0]);
	exit(0);
    default:
	usage(argv[0]);
	exit(2);
    }
  }

  if (optind > argc - 1) {
      /* must have a filename (need to rewind it!)	*/
      usage(argv[0]);
      exit(2);
  }

  for( ; optind <= argc-1; optind++) {

      fd = open(argv[optind],O_RDONLY);
      if (fd < 0) {
	  fprintf(stderr,"%s: file not found \"%s\"\n",argv[0],argv[optind]);
	  exit(2);
      }

      if (force_format_id == NULL) {
	  char 	*ext;

	  ext = floppy_file_extension(argv[optind]);

	  format_id = floppy_format_guesser(fd, ext);
      }

      if (just_guess || report) {
	  int	level = (!report)?1:(!verbose)?2:3;
	  if (machineform) {
	      level = 4;
	  }

	  if (!terse && !machineform) {
	      printf("%s: %s - ",argv[0], argv[optind]);
	  }

	  if (force_format_id == NULL && format_id == NULL) {
	      printf("UNKNOWN FORMAT\n");
	  } else {
	      floppy_format_report((force_format_id==NULL)?format_id:force_format_id,fd,&floppy,level);
	  }

      } else {

	  lseek(fd,(off_t)0,SEEK_SET);

	  if (force_format_id != NULL || format_id != NULL) {
	      if (floppy_format_reader((force_format_id==NULL)?format_id:force_format_id,fd,&floppy) !=0) {
		  fprintf(stderr,"%s: error reading %s format\n",argv[0],format_id);
		  exit(1);
	      }
	  }

	  if (wp) {
	      floppy.write_protect = 0xff;	/* if forcing write protect	*/
	  }

	  if (force_CRC) {
	      floppy_crc_set(&floppy);
	  } else {
	      if (!ignore_CRC) {
		  if (floppy_crc_check(&floppy,1,verbose,argv[0]) != 0) {	/* check the CRCs	*/
		      fprintf(stderr,"%s: CRC error(s).  No output written.  Use -i or -C.\n",argv[0]);
		      exit (2);
		  }
	      }
	  }

	  floppy_sectormap(&floppy);

	  if (finddir) {
	      int	dirtrack = floppy_find_dir_track(&floppy,0);

	      if (dirtrack == floppy.tracks) {
		  printf("No directory found.\n");
	      } else {
		  printf("Directory on physical track %d\n",dirtrack);
	      }
	  }

	  if (dumpsect) {
	      if (dumpphys) {
		  if (track > (floppy.tracks-1)) {
		      fprintf(stderr,"%s: image has only %d tracks\n",argv[0],floppy.tracks);
		      exit(3);
		  }
		  if (sector > (floppy.sectors-1)) {
		      fprintf(stderr,"%s: image has only %d sectors\n",argv[0],floppy.sectors);
		      exit(3);
		  }
	      } else {
		  if (!floppy_find_track(&floppy,track)) {
		      fprintf(stderr,"%s: image doesn't have a logical track %d\n",argv[0],track);
		      exit(3);
		  }
		  if (!floppy_find_sector(&floppy,track,sector)) {
		      fprintf(stderr,"%s: image doesn't have a logical sector %d on track %d\n",
			      argv[0],sector,track);
		      exit(3);
		  }
	      }
	      (void)floppy_dump_sector(stdout,&floppy,track,sector,dumpphys,(inoctal)?1:0);
	  }
      }
      fflush(stdout);
      fflush(stderr);
  }

  exit(0);
}

usage(char *name)
{
    char	buffer[500];

  printf("Usage: %s [option] FILENAME [FILENAMES...]\n\n",name);

  printf("   Prints out interesting things about the given image(s).\n\n");

  printf("   Automagically guesses the FILENAME format\n");
  printf("   which is one of %s.\n", floppy_format_list_string(buffer,FALSE,FALSE));
  printf("   If it doesn't guess right, use the -s fmt option.\n");
  printf("       -T #   ---  set track number for generating reports\n");
  printf("       -S #   ---  set sector number for generating reports\n");
  printf("       -L     ---  interpret sector/track as logical numbers (default to physical)\n");
  printf("       -D     ---  dump a sector, defaults to physical sector 0, track 0\n");
  printf("       -F     ---  find the directory physical track\n");
  printf("\n");
  printf("       -s fmt ---  force given interpretation\n");
  printf("                   (one of %s)\n", floppy_format_list_string(buffer,FALSE,FALSE));
  printf("       -c     ---  just check the disk CRC if possible\n");
  printf("       -C     ---  recalc and write CRC during process\n");
  printf("       -w     ---  treat the disk as write protected\n");
  printf("       -v     ---  be verbose when working\n");
  printf("       -t     ---  be terse when working (used in GUI)\n");
  printf("       -m     ---  output machine readable form (used in GUI)\n");
  printf("       -i     ---  ignore CRC errors\n");
  printf("       -r     ---  long report on given disk\n");
  printf("       -V     ---  print out version number\n");
  printf("       -g     ---  just do a format guess\n\n");
}
